<?php

namespace App\Console\Commands;

use App\Models\News;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class ImportNews extends Command
{
    protected $signature = 'news:import';
    protected $description = 'Import news from JSON file';

    public function handle()
    {
        $json = Storage::disk('public')->get('news.json');
        $news = json_decode($json, true);

        // French to English month mapping
        $frenchToEnglish = [
            'Janvier' => 'January',
            'Février' => 'February',
            'Fevrier' => 'February',
            'Mars' => 'March',
            'Avril' => 'April',
            'Mai' => 'May',
            'Juin' => 'June',
            'Juillet' => 'July',
            'Août' => 'August',
            'Aout' => 'August',
            'Septembre' => 'September',
            'Octobre' => 'October',
            'Novembre' => 'November',
            'Décembre' => 'December',
            'Decembre' => 'December'
        ];

        foreach ($news as $item) {
            $date = $item['date'];

            // Replace French month with English
            foreach ($frenchToEnglish as $french => $english) {
                $date = str_replace($french, $english, $date);
            }

            // Clean up commas and ordinals
            $date = preg_replace('/\b(\d+)(st|nd|rd|th)\b/', '$1', $date);
            $date = str_replace(',', '', $date);

            $this->info("Parsing cleaned date: $date");

            try {
                $parsedDate = Carbon::parse($date);
            } catch (\Exception $e) {
                $this->error("Failed to parse date: $date");
                continue;
            }

            $attachments = [];


            if (isset($item['pdf']) && !empty($item['pdf'])) {
                $attachments[] = [
                    'type' => 'pdf',
                    'url' => $item['pdf'],
                    'filename' => basename($item['pdf'])
                ];
            }

            // Check for image attachment
            if (isset($item['image']) && !empty($item['image'])) {
                $attachments[] = [
                    'type' => 'image',
                    'url' => $item['image'],
                    'filename' => basename($item['image'])
                ];
            }

            News::create([
                'title' => $item['title'],
                'body' => $item['body'],
                'image' => $item['image'],
                'date' => $parsedDate,
                'category' => $item['category'],
                'tags' => $item['tags'],
                'attachments' => $attachments
            ]);
        }


        $this->info('News imported successfully!');
    }
}
