<?php

namespace App\Filament\Resources;

use App\Filament\Resources\NewsResource\Pages;
use App\Models\News;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\Card;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Storage;

class NewsResource extends Resource
{
    protected static ?string $model = News::class;

    protected static ?string $navigationIcon = 'heroicon-o-newspaper';

    protected static ?string $navigationGroup = 'Content Management';

    protected static ?int $navigationSort = 1;


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Card::make()
                    ->schema([
                        TextInput::make('title')
                            ->required()
                            ->maxLength(255),

                        RichEditor::make('body')
                            ->required()
                            ->disableToolbarButtons(['attachFiles']),

                        Forms\Components\Select::make('category')
                            ->required()
                            ->options([
                                'Technology' => 'technology',
                                'esg' => 'esg',
                                'Press Release' => 'Press Release',
                                'Press Articles' => 'Press Articles',
                                'News' => 'News',
                                'Evens' => 'event'
                            ]),

                        TextInput::make('tags'),

                        DatePicker::make('date')->required(),

                        FileUpload::make('image')
                            ->label('News Image')
                            ->disk('public')
                            ->directory('news')
                            ->visibility('public')
                            ->image()
                            ->imagePreviewHeight('200')
                            ->required(fn(?News $record): bool => $record === null)
                            ->preserveFilenames()
                            ->enableOpen()
                            ->enableDownload()
                            ->dehydrated(fn($state) => filled($state))
                            ->deleteUploadedFileUsing(null)
                            ->acceptedFileTypes(['image/jpeg', 'image/png', 'image/webp']),

                        Repeater::make('attachments')
                            ->schema([
                                Forms\Components\Select::make('type')
                                    ->options([
                                        'pdf' => 'PDF Document',
                                        'image' => 'Image',
                                    ])
                                    ->required(),

                                FileUpload::make('url')
                                    ->label('File')
                                    ->disk('public')
                                    ->directory('news/attachments')
                                    ->acceptedFileTypes([
                                        'application/pdf',
                                        'image/jpeg',
                                        'image/png',
                                        'image/webp'
                                    ])
                                    ->enableOpen()
                                    ->enableDownload()
                                    ->preserveFilenames()
                                    ->dehydrated(fn($state) => filled($state))
                                    ->deleteUploadedFileUsing(null),

                                TextInput::make('filename')
                                    ->label('Display Name')
                                    ->required(),
                            ])
                            ->itemLabel(fn(array $state): ?string => $state['filename'] ?? 'Attachment')
                            ->defaultItems(0)
                            ->reorderableWithButtons(),
                    ]),
                ]);
            
    }



    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\ImageColumn::make('image_url')
                    ->disk('public')
                    ->square()
                    ->visibility('public'),
                Tables\Columns\TextColumn::make('date')
                    ->searchable()
                    ->formatStateUsing(function ($state, $record) {
                        if (!$state) return '';
                        // Use the model's accessor to get the properly formatted date
                        return $record->date;
                    }),
                Tables\Columns\TextColumn::make('category')
                    ->badge()
                    ->color(fn(string $state): string => match ($state) {
                        'ESG' => 'success',
                        'Technology' => 'info',
                        'Press Release' => 'warning',
                        'Press Articles' => 'primary',
                        'News' => 'secondary',
                        'Events' => 'danger',
                        default => 'gray',
                    }),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('attachments')
                    ->label('Attachments')
                    ->formatStateUsing(function ($state) {
                        if (empty($state)) return '0 files';
                        $attachments = is_string($state) ? json_decode($state, true) : $state;
                        return is_array($attachments) ? count($attachments) . ' files' : '0 files';
                    }),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('category')
                    ->options(config('news.categories')),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListNews::route('/'),
            'create' => Pages\CreateNews::route('/create'),
            'edit' => Pages\EditNews::route('/{record}/edit'),
        ];
    }
}
