<?php

namespace App\Http\Controllers;

use App\Services\NewsService;
use App\Services\StaticContentIndexer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class SearchController extends Controller
{
    protected $newsService;
    protected $staticContentIndexer;

    public function __construct(NewsService $newsService, StaticContentIndexer $staticContentIndexer)
    {
        $this->newsService = $newsService;
        $this->staticContentIndexer = $staticContentIndexer;
    }

    public function search(Request $request)
    {
        $query = $request->input('query', '');
        
        if (empty($query)) {
            return response()->json([]);
        }

        try {
            // Get news results using the new NewsService
            $newsResults = $this->newsService->searchNews($query);
            $filteredNews = collect($newsResults)
                ->map(function($item) {
                    return [
                        'type' => 'news',
                        'title' => $item->title,
                        'description' => Str::limit(strip_tags($item->body), 200, '...'),
                        'url' => route('news.details', $item->id),
                        'date' => $item->created_at->format('d M Y'),
                        'score' => 1 // Base score for news items
                    ];
                })
                ->take(5)
                ->values()
                ->toArray();

            // Get static page results
            $staticResults = $this->staticContentIndexer->search($query);

            // Format static results to match news results structure
            $formattedStaticResults = collect($staticResults)
                ->map(function($item) {
                    return [
                        'type' => 'page',
                        'title' => $item['title'],
                        'description' => isset($item['matches'][0]['text']) ? Str::limit($item['matches'][0]['text'], 200, '...') : '',
                        'url' => $item['url'],
                        'score' => $item['score'],
                        'matches' => $item['matches']
                    ];
                })
                ->toArray();

            // Combine and sort by score
            $results = array_merge($filteredNews, $formattedStaticResults);
            usort($results, function($a, $b) {
                return $b['score'] <=> $a['score'];
            });

            // Limit to 10 results
            $results = array_slice($results, 0, 10);

            return response()->json($results);
        } catch (\Exception $e) {
            Log::error('Search error: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred during search'], 500);
        }
    }

    public function index()
    {
        try {
            // Index static content
            $this->staticContentIndexer->index();
            return response()->json(['message' => 'Static content indexed successfully']);
        } catch (\Exception $e) {
            Log::error('Indexing error: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred during indexing'], 500);
        }
    }
} 