<?php

namespace App\Services;

use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class StaticContentIndexer
{
    protected $indexedContent = [];
    protected $excludedViews = [
        'layouts.app',
        'includes.head',
        'includes.nav',
        'includes.footer',
        'includes.script',
        'errors.404',
        'errors.500',
        'errors.503',
        'errors.419',
        'errors.403',
        'errors.401',
        'emails.',
        'admin.',
        'components.',
        'layouts.filament',
        '.errors.404',
        '.errors.500',
        '.errors.503',
        '.errors.419',
        '.errors.403',
        '.errors.401'
    ];

    protected $viewToUrlMap = [
        '/about-us' => '/about-us',
        '/our-brand' => '/our-brand',
        '/our-leadership' => '/our-leadership',
        '/our-culture' => '/our-culture',
        '/operation-structure' => '/operation-structure',
        '/innovation' => '/innovation',
        '/csr' => '/csr',
        '/our-reach' => '/our-reach',
        '/foot-prints' => '/foot-prints',
        '/contact-us' => '/contact-us',
        '/news-room' => '/news-room',
        '/phoenix-earth' => '/phoenix-earth',
        '/privacy' => '/privacy',
        '/terms_conditions' => '/terms_conditions',
        '/cookies' => '/cookies',
        '/sustainability' => '/sustainability',
        '/our-heritage' => '/our-heritage',
        'reports/integrated-report' => '/pbg-integrated-reports',
        'investor-relations' => '/investor-relations',
        '/reports/annual-report' => '/reports/annual',
        '/reports/cicl-holdings-arch-report' => '/reports/ciclHoldingsReport',
        '/reports/pbl-arch-report' => '/reports/pblarchiveReport',
        '/reports/picl-holdings-arch-report' => '/reports/piclHoldingsReport',
        '/corporate-governance' => '/corporate-governance',
        '/shareholders' => '/shareholders',
        '/holding-companies' => '/holding-companies',
        '/holding-companies-picl' => '/holding-companies/picl',
        '/holding-companies-cicl' => '/holding-companies/cicl',
        '/non-alcoholic-drink' => '/campaign-non-alcohol',
        '/alcoholic-drink' => '/campaign-alcohol',
        '/welcome' => '/',
        '/general-campaign' => '/general-campaign',
        '/careers' => '/careers',
    ];

    public function index()
    {
        $views = $this->getAllViews();

        $index = [];

        foreach ($views as $view) {
            if ($this->shouldExcludeView($view)) {
                continue;
            }

            $content = $this->getViewContent($view);
            if ($content) {
                $index[] = [
                    'view' => $view,
                    'title' => $this->extractTitle($content),
                    'headings' => $this->extractHeadings($content),
                    'content' => $this->extractContent($content),
                    'url' => $this->getViewUrl($view),
                    'last_updated' => now()
                ];
            }
        }

        // Cache the index for 24 hours
        Cache::put('static_content_index', $index, now()->addHours(24));

        return $index;
    }

    protected function getAllViews()
    {
        $views = [];
        $viewPath = resource_path('views');
        
        $this->scanDirectory($viewPath, $views);
        
        return array_map(function($path) use ($viewPath) {
            return str_replace([$viewPath, '.blade.php', '/'], ['', '', '.'], $path);
        }, $views);
    }

    protected function scanDirectory($directory, &$views)
    {
        $files = scandir($directory);
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }

            $path = $directory . '/' . $file;
            
            if (is_dir($path)) {
                $this->scanDirectory($path, $views);
            } elseif (Str::endsWith($file, '.blade.php')) {
                $views[] = $path;
            }
        }
    }

    protected function shouldExcludeView($view)
    {
        // Remove leading dot if present
        $view = ltrim($view, '.');
        
        foreach ($this->excludedViews as $excluded) {
            if (Str::startsWith($view, $excluded)) {
                return true;
            }
        }
        return false;
    }

    protected function getViewContent($view)
    {
        try {
            // Create a view instance with empty data to prevent undefined variable errors
            $viewInstance = View::make($view, [
                'reports' => [],
                'contacts' => [],
                'news' => [],
                'products' => [],
                'investors' => [],
                'media' => [],
                'careers' => [],
                'sustainability' => [],
                'about' => [],
                'brand' => [],
                'team' => [],
                'history' => [],
                'values' => [],
                'mission' => [],
                'vision' => [],
                'awards' => [],
                'certifications' => [],
                'partners' => [],
                'clients' => [],
                'testimonials' => [],
                'faqs' => [],
                'privacy' => [],
                'terms' => [],
                'cookies' => [],
                'accessibility' => [],
                'sitemap' => [],
                'shareholders' => [],
                'holdingCompanies' => [],
                'corporateGovernance' => [],
                'investorRelations' => [],
                'alcoholicDrinks' => [],
                'nonAlcoholicDrinks' => [],
                'leadership' => [],
                'culture' => [],
                'heritage' => [],
                'footprints' => [],
                'reach' => [],
                'structure' => [],
                'csr' => [],
                'innovation' => [],
                'earth' => [],
                'general-campaign' => [],
            ]);
            
            return $viewInstance->render();
        } catch (\Exception $e) {
            Log::error("Error rendering view {$view}: " . $e->getMessage());
            return null;
        }
    }

    protected function extractTitle($content)
    {
        if (preg_match('/<title>(.*?)<\/title>/i', $content, $matches)) {
            return strip_tags($matches[1]);
        }
        return null;
    }

    protected function extractHeadings($content)
    {
        $headings = [];
        preg_match_all('/<h([1-6])[^>]*>(.*?)<\/h\1>/i', $content, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $text = strip_tags($match[2]);
            if (!empty(trim($text))) {
                $headings[] = [
                    'level' => $match[1],
                    'text' => $text
                ];
            }
        }

        return $headings;
    }

    protected function extractContent($content)
    {
        // Remove script and style tags
        $content = preg_replace('/<script\b[^>]*>(.*?)<\/script>/is', '', $content);
        $content = preg_replace('/<style\b[^>]*>(.*?)<\/style>/is', '', $content);
        
        // Extract text content from paragraphs and other content elements
        $textContent = '';
        if (preg_match_all('/<p[^>]*>(.*?)<\/p>/i', $content, $matches)) {
            $textContent = implode(' ', array_map('strip_tags', $matches[1]));
        }
        
        return $textContent;
    }

    protected function getViewUrl($view)
    {
        // Remove 'index' from the end of the view path
        $view = preg_replace('/\.index$/', '', $view);

        $path = str_replace('.', '/', $view);
    
        if (isset($this->viewToUrlMap[$path])) {
           
            return url($this->viewToUrlMap[$path]);
        }
    
        return url($path);
    }

    public function search($query)
    {
        $index = Cache::get('static_content_index', []);
        $results = [];

        foreach ($index as $item) {
            $score = 0;
            $matches = [];

            // Search in title
            if (stripos($item['title'], $query) !== false) {
                $score += 2;
                $matches[] = [
                    'type' => 'title',
                    'text' => $item['title']
                ];
            }

            // Search in headings
            foreach ($item['headings'] as $heading) {
                if (stripos($heading['text'], $query) !== false) {
                    $score += 1.5;
                    $matches[] = [
                        'type' => 'heading',
                        'level' => $heading['level'],
                        'text' => $heading['text']
                    ];
                }
            }

            // Search in content
            if (stripos($item['content'], $query) !== false) {
                $score += 1;
                $matches[] = [
                    'type' => 'content',
                    'text' => Str::limit($item['content'], 200, '...')
                ];
            }

            if ($score > 0) {
                $results[] = [
                    'type' => 'page',
                    'title' => $item['title'],
                    'url' => $item['url'],
                    'score' => $score,
                    'matches' => $matches
                ];
            }
        }

        // Sort by score
        usort($results, function($a, $b) {
            return $b['score'] <=> $a['score'];
        });

        return $results;
    }
} 