<?php

namespace App\Services;

use App\Models\Stock;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;

class StockService
{
    /**
     * Create a new class instance.
     */
    public function __construct()
    {
        //
    }

    public function getCurrentStockData()
    {
        $cacheKey = 'current_stock_data';
        $cacheDuration = 5;

        $stockData = Cache::get($cacheKey);

        if (!$stockData) {
            $url = 'https://www.stockexchangeofmauritius.com/shareprice/pbl/pblquotes.xml';

            try {
                $response = Http::get($url);

                if ($response->successful()) {
                    $xml = simplexml_load_string($response->body());

                    $timestamp = (string) $xml->timestamp;
                    $time = (string) $xml->timestamp->time;
                    $date = (string) $xml->timestamp->date;
                    $price = (float) $xml->quote->sharepprice;
                    $semdex = (float) $xml->quote->semdex;

                    $stockData = [
                        'timestamp' => $timestamp,
                        'date' => $date,
                        'time' => $time,
                        'price' => $price,
                        'semdex' => $semdex,
                    ];

                    Cache::put($cacheKey, $stockData, $cacheDuration);

                    return $stockData;
                } else {
                    Log::error('Failed to fetch stock data from the API. Response was not successful.');
                }
            } catch (\Exception $e) {
                Log::error('Error occurred while fetching stock data: ' . $e->getMessage());
            }

            // Return fallback data in case of error
            return $this->getFallbackStockData();
        }

        // Return cached stock data if available
        return $stockData;
    }

    /**
     * Provide fallback data if there's an issue fetching the stock data
     *
     * @return array
     */
    private function getFallbackStockData()
    {
        return [
            'timestamp' => '15:45:31',
            'date' => '28-Apr-2025',
            'time' => '15:45:31',
            'price' => 630.00,
            'semdex' => 2387.08,
        ];
    }

    public function getStockDifference()
    {
        // Get the most recent stock price from database
        $oldStockPrice = Stock::latest()->first()->closing_price;

        $stockData = $this->getCurrentStockData();
        $price = $stockData['price'];
        $date = $stockData['date'];

        // Parse and format the date
        $dateTime = \DateTime::createFromFormat('d-M-Y', $date);
        if ($dateTime) {
            $formattedDate = $dateTime->format('Y-m-d');
        } else {
            Log::info("Invalid date format: $date");
        }

        $priceDifference = $price - $oldStockPrice;

        $percentageChange = ($priceDifference / $oldStockPrice) * 100;
        $formattedDifference = number_format($priceDifference, 2);
        $formattedPercentage = number_format($percentageChange, 2);

        $data = [
            'diff' => $formattedDifference,
            'percentageDiff' => $formattedPercentage
        ];

        return $data;
    }

    /**
     * Get real-time stock information for display
     *
     * @return array
     */
    public function getRealTimeStockInfo()
    {
        $stockData = $this->getCurrentStockData();
        
        // Get the most recent database record for comparison
        $latestDbRecord = Stock::latest()->first();
        $previousPrice = $latestDbRecord ? (float) $latestDbRecord->closing_price : 0;
        
        $currentPrice = (float) $stockData['price'];
        
        // Calculate price change
        $priceChange = $currentPrice - $previousPrice;
        $percentageChange = $previousPrice > 0 ? ($priceChange / $previousPrice) * 100 : 0;
        
        // Format the values
        $formattedPrice = number_format($currentPrice, 2);
        $formattedChange = number_format($priceChange, 2);
        $formattedPercentage = number_format($percentageChange, 2);
        
        // Add sign to change values
        $signedChange = $priceChange >= 0 ? '+' . $formattedChange : $formattedChange;
        $signedPercentage = $percentageChange >= 0 ? '+' . $formattedPercentage : $formattedPercentage;
        
        return [
            'current_price' => $formattedPrice,
            'price_change' => $signedChange,
            'percentage_change' => $signedPercentage,
            'is_positive' => $priceChange >= 0,
            'last_updated' => $stockData['date'] . ' ' . $stockData['time'],
            'raw_price' => $currentPrice,
            'raw_change' => $priceChange,
            'raw_percentage' => $percentageChange
        ];
    }

    /**
     * Get comprehensive stock and financial data for shareholders page
     *
     * @return array
     */
    public function getShareholdersPageData()
    {
        $stockData = $this->getCurrentStockData();
        
        // Get the most recent database record for comparison
        $latestDbRecord = Stock::latest()->first();
        $previousPrice = $latestDbRecord ? (float) $latestDbRecord->closing_price : 0;
        
        $currentPrice = (float) $stockData['price'];
        
        // Calculate price change
        $priceChange = $currentPrice - $previousPrice;
        $percentageChange = $previousPrice > 0 ? ($priceChange / $previousPrice) * 100 : 0;
        
        // Format the values
        $formattedPrice = number_format($currentPrice, 2);
        $formattedChange = number_format($priceChange, 2);
        $formattedPercentage = number_format($percentageChange, 2);
        
        // Add sign to change values
        $signedChange = $priceChange >= 0 ? '+' . $formattedChange : $formattedChange;
        $signedPercentage = $percentageChange >= 0 ? '+' . $formattedPercentage : $formattedPercentage;
        
        // Financial metrics (these would ideally come from a financial data API or database)
        $financialMetrics = [
            'total_assets' => [
                '2023' => 9.0e9,
                '2024' => 10.1e9
            ],
            'turnover' => [
                '2023' => 10.6e9,
                '2024' => 12.2e9
            ],
            'operating_profits' => [
                '2023' => 0.9e9,
                '2024' => 1.4e9
            ],
            'capital_investment' => [
                '2023' => 407e6,
                '2024' => 581e6
            ],
            'shareholders_fund' => [
                '2023' => 6.1e9,
                '2024' => 6.8e9
            ],
            'ebitda' => [
                '2023' => 1.4e9,
                '2024' => 1.9e9
            ]
        ];
        
        // Share price trend data (last 3 years)
        $sharePriceTrend = [
            'labels' => ['Jan 2022', 'Sep 2023', 'Jan 2024', 'Sep 2024'],
            'data' => [69.5, 54, 56.5, $currentPrice] // Use current price for latest data point
        ];
        
        return [
            'stock_info' => [
                'current_price' => $formattedPrice,
                'price_change' => $signedChange,
                'percentage_change' => $signedPercentage,
                'is_positive' => $priceChange >= 0,
                'last_updated' => $stockData['date'] . ' ' . $stockData['time'],
                'raw_price' => $currentPrice,
                'raw_change' => $priceChange,
                'raw_percentage' => $percentageChange,
                'previous_close' => number_format($previousPrice, 2),
                'earning_per_share' => number_format($currentPrice * 0.15, 2) // Estimated EPS
            ],
            'financial_metrics' => $financialMetrics,
            'share_price_trend' => $sharePriceTrend
        ];
    }
}
